<?php
/**
 * Script para verificar novas OS e enviar notificações push
 * Este script deve ser executado via cron job a cada X minutos
 */

header('Content-Type: application/json');

// Configurações
$tokensFile = __DIR__ . '/tokens-push.json';
$ultimaVerificacaoFile = __DIR__ . '/ultima-verificacao-os.json';
$serviceAccountFile = __DIR__ . '/fcm-service-account.json';

require(__DIR__ . DIRECTORY_SEPARATOR . 'WebserviceClient.php');
require(__DIR__ . DIRECTORY_SEPARATOR . 'enviar-push-fcm-v1.php');

function base64UrlEncode($data)
{
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function obterAccessTokenFCM($serviceAccountFile)
{
    static $cachedToken = null;
    static $cachedExpiry = 0;

    if ($cachedToken && $cachedExpiry > time() + 300) {
        return $cachedToken;
    }

    if (!file_exists($serviceAccountFile)) {
        error_log('Arquivo de service account não encontrado: ' . $serviceAccountFile);
        return null;
    }

    $creds = json_decode(file_get_contents($serviceAccountFile), true);
    if (!$creds || empty($creds['private_key']) || empty($creds['client_email'])) {
        error_log('Credenciais inválidas no service account.');
        return null;
    }

    $now = time();
    $jwtHeader = base64UrlEncode(json_encode(['alg' => 'RS256', 'typ' => 'JWT']));
    $jwtClaim = base64UrlEncode(json_encode([
        'iss' => $creds['client_email'],
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        'aud' => $creds['token_uri'],
        'iat' => $now,
        'exp' => $now + 3600
    ]));

    $unsignedJwt = $jwtHeader . '.' . $jwtClaim;
    $signature = '';
    if (!openssl_sign($unsignedJwt, $signature, $creds['private_key'], 'sha256WithRSAEncryption')) {
        error_log('Falha ao assinar JWT para FCM.');
        return null;
    }

    $signedJwt = $unsignedJwt . '.' . base64UrlEncode($signature);

    $postFields = http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $signedJwt
    ]);

    $ch = curl_init('https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/x-www-form-urlencoded'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        error_log('Erro cURL ao obter access token FCM: ' . curl_error($ch));
    }
    curl_close($ch);

    if ($httpCode !== 200) {
        error_log('Falha ao obter access token FCM. HTTP ' . $httpCode . ' Resposta: ' . $response);
        return null;
    }

    $tokenResponse = json_decode($response, true);
    if (!isset($tokenResponse['access_token'])) {
        error_log('Resposta inválida ao obter access token: ' . $response);
        return null;
    }

    $cachedToken = $tokenResponse['access_token'];
    $cachedExpiry = $now + ($tokenResponse['expires_in'] ?? 3500);

    return $cachedToken;
}

$host = 'https://app.core3.com.br/webservice/v1';
$token = '91:e6dd49bd36efd72ac97b5e2abeb2fb09dbb081a597a7b2dd19d82926bb2f73a9';
$selfSigned = true;
$api = new IXCsoft\WebserviceClient($host, $token, $selfSigned);

// Ler tokens push salvos
$tokens = [];
if (file_exists($tokensFile)) {
    $content = file_get_contents($tokensFile);
    $tokens = json_decode($content, true) ?: [];
}

if (empty($tokens)) {
    echo json_encode(['success' => false, 'message' => 'Nenhum token push encontrado']);
    exit;
}

// Ler última verificação
$ultimaVerificacao = [];
if (file_exists($ultimaVerificacaoFile)) {
    $content = file_get_contents($ultimaVerificacaoFile);
    $ultimaVerificacao = json_decode($content, true) ?: [];
}

$notificacoesEnviadas = 0;
$erros = [];

foreach ($tokens as $idTecnico => $tokenData) {
    try {
        $pushToken = $tokenData['push_token'] ?? null;
        if (!$pushToken) {
            continue;
        }

        $tipoToken = $tokenData['tipo'] ?? 'fcm';
        // Aceitar apenas tokens FCM
        if ($tipoToken !== 'fcm') {
            continue;
        }
        
        // Buscar OS do técnico
        $params = array(
            'qtype' => 'su_oss_chamado.id_tecnico',
            'query' => $idTecnico,
            'oper' => '=',
            'rp' => '100',
            'sortname' => 'su_oss_chamado.data_abertura',
            'sortorder' => 'desc'
        );
        
        $api->get('su_oss_chamado', $params);
        $retorno = $api->getRespostaConteudo(false);
        $dados = json_decode($retorno, true);
        
        if (!isset($dados['registros']) || count($dados['registros']) === 0) {
            continue;
        }
        
        // Filtrar apenas OS com status AG (Agendamento)
        $osAgendamento = array_filter($dados['registros'], function($os) {
            return isset($os['status']) && $os['status'] === 'AG';
        });
        
        if (count($osAgendamento) === 0) {
            continue;
        }
        
        // Reindexar array
        $osAgendamento = array_values($osAgendamento);
        
        // Verificar última OS conhecida
        $ultimaOSConhecida = $ultimaVerificacao[$idTecnico] ?? null;
        $novasOS = [];
        
        foreach ($osAgendamento as $os) {
            $osId = $os['id'] ?? null;
            $dataAbertura = $os['data_abertura'] ?? null;
            
            // Se não temos última OS registrada, registrar esta e continuar
            if (!$ultimaOSConhecida) {
                $ultimaVerificacao[$idTecnico] = [
                    'id' => $osId,
                    'data_abertura' => $dataAbertura
                ];
                continue;
            }
            
            // Se esta OS é mais recente que a última conhecida, é nova
            if ($osId != $ultimaVerificacao[$idTecnico]['id'] && 
                $dataAbertura > $ultimaVerificacao[$idTecnico]['data_abertura']) {
                $novasOS[] = $os;
            }
        }
        
        // Se encontrou novas OS, enviar notificação
        if (count($novasOS) > 0) {
            $qtdNovas = count($novasOS);
            
            // Buscar nome do cliente da primeira OS nova
            $primeiraOS = $novasOS[0];
            $idCliente = $primeiraOS['id_cliente'] ?? null;
            $nomeCliente = 'Cliente';
            
            if ($idCliente) {
                // Buscar razão social do cliente
                $paramsCliente = array(
                    'qtype' => 'cliente.id',
                    'query' => $idCliente,
                    'oper' => '=',
                    'page' => '1',
                    'rp' => '1',
                    'sortname' => 'cliente.id',
                    'sortorder' => 'desc'
                );
                
                $api->get('cliente', $paramsCliente);
                $retornoCliente = $api->getRespostaConteudo(false);
                $dadosCliente = json_decode($retornoCliente, true);
                
                if (isset($dadosCliente['registros'][0]['razao'])) {
                    $nomeCliente = $dadosCliente['registros'][0]['razao'];
                }
            }
            
            // Montar mensagem personalizada
            if ($qtdNovas === 1) {
                $mensagem = "Nova OS: Cliente: {$nomeCliente} atribuída a você";
            } else {
                $mensagem = "{$qtdNovas} novas OS atribuídas a você";
            }
            
            // Enviar push notification via FCM v1
            $resultado = enviarPushFCMv1(
                $pushToken,
                'Nova OS Disponível',
                $mensagem,
                [
                    'tipo' => 'nova_os',
                    'qtd' => (string)$qtdNovas
                ]
            );
            
            if ($resultado['success']) {
                $notificacoesEnviadas++;
                // Atualizar última OS conhecida
                $primeiraNova = $novasOS[0];
                $ultimaVerificacao[$idTecnico] = [
                    'id' => $primeiraNova['id'] ?? null,
                    'data_abertura' => $primeiraNova['data_abertura'] ?? null
                ];
            } else {
                $erroMsg = $resultado['message'] ?? 'Erro desconhecido';
                $erroHttp = $resultado['http_code'] ?? 'N/A';
                $erroResp = $resultado['response'] ?? '';
                $erros[] = "Erro ao enviar push para técnico {$idTecnico}: {$erroMsg} (HTTP {$erroHttp}) - {$erroResp}";
            }
        }
        
    } catch (Exception $e) {
        $erros[] = "Erro ao processar técnico {$idTecnico}: " . $e->getMessage();
    }
}

// Salvar última verificação
file_put_contents($ultimaVerificacaoFile, json_encode($ultimaVerificacao, JSON_PRETTY_PRINT));

echo json_encode([
    'success' => true,
    'notificacoes_enviadas' => $notificacoesEnviadas,
    'erros' => $erros,
    'timestamp' => date('Y-m-d H:i:s')
]);
?>

