<?php
/**
 * Script de Desativação Automática de Recorrência - Contratos Bloqueados
 * 
 * Este script deve ser configurado como tarefa cron para executar automaticamente
 * e desativar recorrências de contratos com status de bloqueio (CM/CA).
 * 
 * Uso via cron:
 * Executar a cada 6 horas
 * Exemplo: 0 6 * * * /usr/bin/php /caminho/para/teste_bloqueio.php
 * 
 * @author Sistema Core3Chip
 * @version 2.0
 * @date 2025-01-30
 */

// Configurações de conexão com o banco MySQL
$host = '192.168.2.129';
$user = 'root';
$password = 'C0r3@2018';
$database = 'ixcprovedor';

// Configurações de log
$logFile = __DIR__ . '/logs/desativacao_recorrencia_' . date('Y-m-d') . '.log';
$logDir = __DIR__ . '/logs';

// Criar diretório de logs se não existir
if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

// Função para escrever logs
function writeLog($message, $type = 'INFO') {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[{$timestamp}] [{$type}] {$message}" . PHP_EOL;
    
    // Escrever no arquivo de log
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    
    // Também exibir no console se executado via linha de comando
    if (php_sapi_name() === 'cli') {
        echo $logMessage;
    }
}

// Função para desativar recorrência via API
function desativarRecorrencia($accountId, $contratoId) {
    writeLog("Iniciando desativação de recorrência - Contrato: {$contratoId}, AccountId: {$accountId}");
    
    try {
        // Payload para desativar recorrência
        $putBody = json_encode([
            'enabled' => false,
            'recurrenceType' => 'BILLET'
        ]);
        
        // Tentar métodos em sequência (PUT, PATCH, POST)
        $metodos = ['PUT', 'PATCH', 'POST'];
        $sucesso = false;
        $metodoUsado = '';
        $resposta = '';
        
        foreach ($metodos as $metodo) {
            writeLog("Tentando método HTTP: {$metodo}");
            
            try {
                // Preparar contexto da requisição
                $context = stream_context_create([
                    'http' => [
                        'method' => $metodo,
                        'header' => [
                            'Content-Type: application/json',
                            'Accept: */*',
                            'Content-Length: ' . strlen($putBody)
                        ],
                        'content' => $putBody,
                        'timeout' => 30
                    ]
                ]);
                
                // URL da API
                $url = "http://localhost:8000/proxy-chip.php?rota=" . urlencode("/api/Subscription/{$accountId}/recurrence");
                
                // Fazer a requisição
                $response = file_get_contents($url, false, $context);
                
                if ($response === false) {
                    writeLog("Erro na requisição HTTP com método {$metodo}", 'ERROR');
                    continue;
                }
                
                // Decodificar resposta
                $obj = json_decode($response, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $obj = $response; // Usar resposta como texto se não for JSON válido
                }
                
                // Verificar se não é 405 (Method Not Allowed)
                if (!(is_array($obj) && isset($obj['http_code']) && $obj['http_code'] == 405)) {
                    $sucesso = true;
                    $metodoUsado = $metodo;
                    $resposta = $obj;
                    writeLog("Sucesso com método {$metodo} - Resposta: " . json_encode($obj));
                    break;
                } else {
                    writeLog("Método {$metodo} retornou 405 (Method Not Allowed)");
                }
                
            } catch (Exception $e) {
                writeLog("Erro com método {$metodo}: " . $e->getMessage(), 'ERROR');
            }
        }
        
        if ($sucesso) {
            writeLog("✅ Recorrência desativada com sucesso - Contrato: {$contratoId}, AccountId: {$accountId}, Método: {$metodoUsado}");
            return [
                'success' => true,
                'contrato_id' => $contratoId,
                'account_id' => $accountId,
                'metodo' => $metodoUsado,
                'resposta' => $resposta
            ];
        } else {
            writeLog("❌ Falha ao desativar recorrência - Contrato: {$contratoId}, AccountId: {$accountId} - Todos os métodos falharam", 'ERROR');
            return [
                'success' => false,
                'contrato_id' => $contratoId,
                'account_id' => $accountId,
                'erro' => 'Todos os métodos HTTP falharam'
            ];
        }
        
    } catch (Exception $e) {
        writeLog("❌ Erro geral ao desativar recorrência - Contrato: {$contratoId}, AccountId: {$accountId} - Erro: " . $e->getMessage(), 'ERROR');
        return [
            'success' => false,
            'contrato_id' => $contratoId,
            'account_id' => $accountId,
            'erro' => $e->getMessage()
        ];
    }
}

// Função principal
function main() {
    global $host, $user, $password, $database;
    
    writeLog("🚀 INICIANDO EXECUÇÃO DO SCRIPT DE DESATIVAÇÃO DE RECORRÊNCIA");
    writeLog("Configurações: Host={$host}, Database={$database}");
    
    try {
        // Conectar ao banco MySQL
        writeLog("Conectando ao banco MySQL...");
        $pdo = new PDO("mysql:host={$host};dbname={$database};charset=utf8", $user, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        writeLog("✅ Conexão com banco estabelecida com sucesso");
        
        // Query SQL para buscar os contratos
        $sql = "SELECT cc.`id`, cc.`imp_obs`, cc.`status_internet` 
                FROM cliente_contrato cc 
                WHERE cc.`imp_obs` IS NOT NULL 
                AND cc.`imp_obs` != '' 
                AND cc.`status_internet` IN ('CM', 'CA')";
        
        writeLog("Executando query: " . $sql);
        
        // Executar a query
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $totalContratos = count($resultados);
        writeLog("📊 Total de contratos encontrados: {$totalContratos}");
        
        if ($totalContratos == 0) {
            writeLog("ℹ️ Nenhum contrato encontrado para processar");
            return;
        }
        
        // Estatísticas por status
        $cmCount = 0;
        $caCount = 0;
        foreach ($resultados as $contrato) {
            if ($contrato['status_internet'] === 'CM') {
                $cmCount++;
            } elseif ($contrato['status_internet'] === 'CA') {
                $caCount++;
            }
        }
        
        writeLog("📈 Estatísticas: CM={$cmCount}, CA={$caCount}");
        
        // Processar cada contrato
        $sucessos = 0;
        $falhas = 0;
        $resultadosDetalhados = [];
        
        writeLog("🔄 Iniciando processamento dos contratos...");
        
        foreach ($resultados as $index => $contrato) {
            $numero = $index + 1;
            writeLog("Processando contrato {$numero}/{$totalContratos} - ID: {$contrato['id']}, Status: {$contrato['status_internet']}");
            
            $resultado = desativarRecorrencia($contrato['imp_obs'], $contrato['id']);
            $resultadosDetalhados[] = $resultado;
            
            if ($resultado['success']) {
                $sucessos++;
            } else {
                $falhas++;
            }
            
            // Pequena pausa entre requisições para não sobrecarregar a API
            if ($index < $totalContratos - 1) {
                sleep(1);
            }
        }
        
        // Resumo final
        writeLog("🏁 PROCESSAMENTO CONCLUÍDO!");
        writeLog("📊 RESUMO FINAL:");
        writeLog("   ✅ Sucessos: {$sucessos}");
        writeLog("   ❌ Falhas: {$falhas}");
        writeLog("   📈 Total processado: {$totalContratos}");
        writeLog("   📊 Taxa de sucesso: " . round(($sucessos / $totalContratos) * 100, 2) . "%");
        
        // Salvar resultados detalhados em arquivo JSON
        $resultadoFile = __DIR__ . '/logs/resultado_' . date('Y-m-d_H-i-s') . '.json';
        file_put_contents($resultadoFile, json_encode([
            'timestamp' => date('Y-m-d H:i:s'),
            'total_contratos' => $totalContratos,
            'sucessos' => $sucessos,
            'falhas' => $falhas,
            'resultados' => $resultadosDetalhados
        ], JSON_PRETTY_PRINT));
        
        writeLog("💾 Resultados detalhados salvos em: {$resultadoFile}");
        
    } catch (PDOException $e) {
        writeLog("❌ ERRO DE CONEXÃO COM O BANCO: " . $e->getMessage(), 'ERROR');
        writeLog("Código de erro: " . $e->getCode(), 'ERROR');
    } catch (Exception $e) {
        writeLog("❌ ERRO GERAL: " . $e->getMessage(), 'ERROR');
    }
    
    writeLog("🏁 FIM DA EXECUÇÃO DO SCRIPT");
    writeLog("=" . str_repeat("=", 80));
}

// Executar função principal
main();
?>
