<?php
/**
 * Teste do Sistema de Mapeamento de Planos
 * 
 * Este arquivo testa todas as funcionalidades do sistema de mapeamento
 * entre códigos de planos da Boramvno e IXC.
 */

// Definir constante para acesso ao sistema
define('SYSTEM_ACCESS', true);

// Incluir o sistema de mapeamento
require_once 'mapeamento-planos.php';

// Incluir configurações do IXC
require_once 'confApi.php';

echo "<h1>🧪 Teste do Sistema de Mapeamento de Planos</h1>\n";
echo "<hr>\n\n";

// Teste 1: Verificar se o mapeamento foi carregado
echo "<h2>1. Verificação do Mapeamento</h2>\n";
$mapeamento = obterMapeamentoCompleto();
echo "<p><strong>Total de planos mapeados:</strong> " . count($mapeamento) . "</p>\n";

// Teste 2: Listar todos os códigos Boramvno
echo "<h2>2. Códigos Boramvno Disponíveis</h2>\n";
$codigosBoramvno = MapeamentoPlanos::obterCodigosBoramvno();
echo "<p><strong>Códigos Boramvno:</strong></p>\n";
echo "<ul>\n";
foreach ($codigosBoramvno as $codigo) {
    echo "<li><code>$codigo</code></li>\n";
}
echo "</ul>\n";

// Teste 3: Listar todos os códigos IXC
echo "<h2>3. Códigos IXC Disponíveis</h2>\n";
$codigosIXC = MapeamentoPlanos::obterCodigosIXC();
echo "<p><strong>Códigos IXC:</strong></p>\n";
echo "<ul>\n";
foreach ($codigosIXC as $codigo) {
    echo "<li><code>$codigo</code></li>\n";
}
echo "</ul>\n";

// Teste 4: Testar conversões específicas
echo "<h2>4. Teste de Conversões</h2>\n";
$testeCodigos = [
    '90206972-6c15-4b67-afaa-559fd80ab690', // Deve retornar 3543
    '1d35c399-b945-4193-b575-3c6c1b7dff27', // Deve retornar 3541
    'abfe54df-8105-46cd-95f9-855d89076ddb', // Deve retornar 3523
    'codigo-inexistente' // Deve retornar null
];

echo "<table border='1' style='border-collapse: collapse; width: 100%;'>\n";
echo "<tr style='background-color: #f0f0f0;'>\n";
echo "<th>Código Boramvno</th>\n";
echo "<th>Código IXC</th>\n";
echo "<th>Status</th>\n";
echo "</tr>\n";

foreach ($testeCodigos as $codigoBoramvno) {
    $codigoIXC = obterCodigoIXC($codigoBoramvno);
    $status = $codigoIXC ? '✅ Encontrado' : '❌ Não encontrado';
    $codigoIXCDisplay = $codigoIXC ?: 'N/A';
    
    echo "<tr>\n";
    echo "<td><code>$codigoBoramvno</code></td>\n";
    echo "<td><code>$codigoIXCDisplay</code></td>\n";
    echo "<td>$status</td>\n";
    echo "</tr>\n";
}
echo "</table>\n";

// Teste 5: Testar informações completas dos planos
echo "<h2>5. Informações Completas dos Planos</h2>\n";
$planoTeste = obterInformacoesPlano('7ef69094-030b-4c27-aa3b-85ab62074d90');
if ($planoTeste) {
    echo "<p><strong>Plano encontrado:</strong></p>\n";
    echo "<ul>\n";
    echo "<li><strong>Código IXC:</strong> {$planoTeste['codigo_ixc']}</li>\n";
    echo "<li><strong>Nome:</strong> {$planoTeste['nome_plano']}</li>\n";
    echo "<li><strong>Tipo:</strong> {$planoTeste['tipo_plano']}</li>\n";
    echo "<li><strong>Valor:</strong> R$ {$planoTeste['valor_plano']}</li>\n";
    echo "<li><strong>Duração:</strong> {$planoTeste['duracao_plano']} dias</li>\n";
    echo "<li><strong>Descrição:</strong> {$planoTeste['descricao']}</li>\n";
    echo "</ul>\n";
} else {
    echo "<p>❌ <strong>Erro:</strong> Plano não encontrado</p>\n";
}

// Teste 6: Verificar existência de planos
echo "<h2>6. Verificação de Existência</h2>\n";
$planosParaVerificar = [
    '90206972-6c15-4b67-afaa-559fd80ab690',
    '1d35c399-b945-4193-b575-3c6c1b7dff27',
    'codigo-inexistente'
];

echo "<ul>\n";
foreach ($planosParaVerificar as $codigo) {
    $existe = planoExiste($codigo);
    $status = $existe ? '✅ Existe' : '❌ Não existe';
    echo "<li><code>$codigo</code>: $status</li>\n";
}
echo "</ul>\n";

// Teste 7: Estatísticas do mapeamento
echo "<h2>7. Estatísticas do Mapeamento</h2>\n";
$estatisticas = MapeamentoPlanos::obterEstatisticas();
echo "<ul>\n";
echo "<li><strong>Total de planos:</strong> {$estatisticas['total_planos']}</li>\n";
echo "<li><strong>Tipos disponíveis:</strong></li>\n";
echo "<ul>\n";
foreach ($estatisticas['tipos_disponiveis'] as $tipo => $quantidade) {
    echo "<li>$tipo: $quantidade planos</li>\n";
}
echo "</ul>\n";
echo "<li><strong>Valor médio:</strong> R$ " . number_format($estatisticas['valor_medio'], 2, ',', '.') . "</li>\n";
echo "<li><strong>Valor mínimo:</strong> R$ " . number_format($estatisticas['valor_minimo'], 2, ',', '.') . "</li>\n";
echo "<li><strong>Valor máximo:</strong> R$ " . number_format($estatisticas['valor_maximo'], 2, ',', '.') . "</li>\n";
echo "</ul>\n";

// Teste 8: Validação do mapeamento
echo "<h2>8. Validação do Mapeamento</h2>\n";
$erros = MapeamentoPlanos::validarMapeamento();
if (empty($erros)) {
    echo "<p>✅ <strong>Mapeamento válido:</strong> Nenhum erro encontrado</p>\n";
} else {
    echo "<p>❌ <strong>Erros encontrados:</strong></p>\n";
    echo "<ul>\n";
    foreach ($erros as $erro) {
        echo "<li>$erro</li>\n";
    }
    echo "</ul>\n";
}

// Teste 9: Busca por código IXC
echo "<h2>9. Busca por Código IXC</h2>\n";
$codigosIXCParaBuscar = ['3543', '3523', '9999'];
echo "<ul>\n";
foreach ($codigosIXCParaBuscar as $codigoIXC) {
    $resultado = MapeamentoPlanos::buscarPorCodigoIXC($codigoIXC);
    if ($resultado) {
        echo "<li><strong>IXC $codigoIXC:</strong> Encontrado no plano Boramvno <code>{$resultado['codigo_boramvno']}</code></li>\n";
    } else {
        echo "<li><strong>IXC $codigoIXC:</strong> ❌ Não encontrado</li>\n";
    }
}
echo "</ul>\n";

// Teste 10: Verificar configurações do IXC
echo "<h2>10. Configurações do IXC</h2>\n";
if (isset($api) && $api instanceof IXCsoft\WebserviceClient) {
    echo "<p>✅ <strong>Cliente IXC:</strong> Configurado corretamente</p>\n";
    echo "<ul>\n";
    echo "<li><strong>Host:</strong> " . getConfig('host') . "</li>\n";
    echo "<li><strong>Token:</strong> " . substr(getConfig('token'), 0, 10) . "...</li>\n";
    echo "<li><strong>Self Signed:</strong> " . (getConfig('selfSigned') ? 'Sim' : 'Não') . "</li>\n";
    echo "<li><strong>Timeout:</strong> " . getConfig('timeout') . "s</li>\n";
    echo "</ul>\n";
} else {
    echo "<p>❌ <strong>Erro:</strong> Cliente IXC não configurado</p>\n";
}

// Teste 11: Exportar mapeamento para JSON
echo "<h2>11. Exportação para JSON</h2>\n";
$json = MapeamentoPlanos::exportarParaJSON();
echo "<p><strong>JSON gerado:</strong></p>\n";
echo "<pre style='background-color: #f5f5f5; padding: 10px; border-radius: 5px; overflow-x: auto;'>\n";
echo htmlspecialchars(substr($json, 0, 500)) . "...\n";
echo "</pre>\n";

echo "<hr>\n";
echo "<h2>🎯 Resumo dos Testes</h2>\n";
echo "<p><strong>Status:</strong> Sistema de mapeamento funcionando corretamente!</p>\n";
echo "<p><strong>Total de planos:</strong> " . count($mapeamento) . "</p>\n";
echo "<p><strong>Configuração IXC:</strong> " . (isset($api) ? '✅ OK' : '❌ Erro') . "</p>\n";
echo "<p><strong>Validação:</strong> " . (empty($erros) ? '✅ OK' : '❌ Erros encontrados') . "</p>\n";

echo "<p><em>Teste concluído em: " . date('d/m/Y H:i:s') . "</em></p>\n";
?>
