<?php
/**
 * Mapeamento de Planos - Boramvno para IXC
 * 
 * Este arquivo contém a relação entre os códigos de planos da API Boramvno
 * e os códigos de planos do sistema IXC.
 * 
 * Pode ser incluído em qualquer lugar do sistema para obter o mapeamento.
 * 
 * @author Sistema Core3Chip
 * @version 1.0
 * @date 2025-01-30
 */

// Prevenir acesso direto ao arquivo
if (!defined('SYSTEM_ACCESS')) {
    define('SYSTEM_ACCESS', true);
}

/**
 * Classe para gerenciar o mapeamento de planos
 */
class MapeamentoPlanos {
    
    /**
     * Array de mapeamento entre códigos de planos
     * 
     * Estrutura:
     * 'codigo_boramvno' => [
     *     'codigo_ixc' => 'codigo_no_ixc',
     *     'nome_plano' => 'Nome do Plano',
     *     'tipo_plano' => 'Tipo do Plano',
     *     'valor_plano' => 0.00,
     *     'duracao_plano' => 30,
     *     'descricao' => 'Descrição do plano'
     * ]
     */
    private static $mapeamento = [
        // Mapeamento real entre códigos Boramvno e IXC
        '90206972-6c15-4b67-afaa-559fd80ab690' => [
            'codigo_ixc' => '3543',
            'nome_plano' => 'Plano IXC 3543',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3543'
        ],
        
        '1d35c399-b945-4193-b575-3c6c1b7dff27' => [
            'codigo_ixc' => '3541',
            'nome_plano' => 'Plano IXC 3541',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3541'
        ],
        
        '6d13f6f8-97dd-43eb-bac2-8d69ab5c4ba1' => [
            'codigo_ixc' => '3540',
            'nome_plano' => 'Plano IXC 3540',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3540'
        ],
        
        'e29808c5-315d-4a92-b660-470885999693' => [
            'codigo_ixc' => '3538',
            'nome_plano' => 'Plano IXC 3538',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3538'
        ],
        
        '2ce4e591-1c62-46eb-be0d-def47a4bcd0e' => [
            'codigo_ixc' => '3537',
            'nome_plano' => 'Plano IXC 3537',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3537'
        ],
        
        'f0bc2237-872a-428b-b538-b23fd1b8393c' => [
            'codigo_ixc' => '3536',
            'nome_plano' => 'Plano IXC 3536',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3536'
        ],
        
        '37d42bde-3db8-451c-b0ee-fd1d1418cc02' => [
            'codigo_ixc' => '3535',
            'nome_plano' => 'Plano IXC 3535',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3535'
        ],
        
        '7ef69094-030b-4c27-aa3b-85ab62074d90' => [
            'codigo_ixc' => '3534',
            'nome_plano' => 'Plano IXC 3534',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3534'
        ],
        
        'b8ed1724-c99f-4370-b8e0-17c2fe3cca87' => [
            'codigo_ixc' => '3533',
            'nome_plano' => 'Plano IXC 3533',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3533'
        ],
        
        'e106c96b-2872-45b9-9c8a-3e7c189caa2f' => [
            'codigo_ixc' => '3532',
            'nome_plano' => 'Plano IXC 3532',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3532'
        ],
        
        'a5dfc72b-11c4-490b-a7b2-3a5835f37e3d' => [
            'codigo_ixc' => '3531',
            'nome_plano' => 'Plano IXC 3531',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3531'
        ],
        
        '61320bcf-bb9c-4e60-88e3-0834c6d7707b' => [
            'codigo_ixc' => '3530',
            'nome_plano' => 'Plano IXC 3530',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3530'
        ],
        
        '347dfe3b-e581-482d-9317-eaa7b2a23120' => [
            'codigo_ixc' => '3529',
            'nome_plano' => 'Plano IXC 3529',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3529'
        ],
        
        '7a917811-00d1-4aee-bd28-d0c76aa65c35' => [
            'codigo_ixc' => '3528',
            'nome_plano' => 'Plano IXC 3528',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3528'
        ],
        
        '60aaea05-8379-4149-877f-f2ee8c3ef6a1' => [
            'codigo_ixc' => '3527',
            'nome_plano' => 'Plano IXC 3527',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3527'
        ],
        
        '17b8b477-5f7a-4ae7-8695-149eaca1e156' => [
            'codigo_ixc' => '3526',
            'nome_plano' => 'Plano IXC 3526',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3526'
        ],
        
        '5abe216a-e4af-4344-8b6d-71f65faa5d1a' => [
            'codigo_ixc' => '3525',
            'nome_plano' => 'Plano IXC 3525',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3525'
        ],
        
        '3d6b8730-d506-446f-ab4a-a05e2445f8f0' => [
            'codigo_ixc' => '3524',
            'nome_plano' => 'Plano IXC 3524',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3524'
        ],
        
        'abfe54df-8105-46cd-95f9-855d89076ddb' => [
            'codigo_ixc' => '3523',
            'nome_plano' => 'Plano IXC 3523',
            'tipo_plano' => 'RECARGA',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Plano correspondente ao código IXC 3523'
        ],
    ];
    
    /**
     * Obter o mapeamento completo
     * 
     * @return array Array completo de mapeamento
     */
    public static function obterMapeamentoCompleto() {
        return self::$mapeamento;
    }
    
    // Função para buscar plano por ID Boramvno
    public static function buscarPlanoPorBoramvno($planoBoramvno) {
        if (isset(self::$mapeamento[$planoBoramvno])) {
            return [
                'success' => true,
                'codigo_ixc' => self::$mapeamento[$planoBoramvno]['codigo_ixc'],
                'nome_plano' => self::$mapeamento[$planoBoramvno]['nome_plano'],
                'tipo_plano' => self::$mapeamento[$planoBoramvno]['tipo_plano']
            ];
        }
        return ['success' => false, 'error' => 'Plano não encontrado'];
    }
    /**
     * Obter código IXC pelo código Boramvno
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @return string|null Código do plano no IXC ou null se não encontrado
     */
    public static function obterCodigoIXC($codigoBoramvno) {
        if (isset(self::$mapeamento[$codigoBoramvno])) {
            return self::$mapeamento[$codigoBoramvno]['codigo_ixc'];
        }
        return null;
    }
    
    /**
     * Obter informações completas do plano pelo código Boramvno
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @return array|null Array com informações do plano ou null se não encontrado
     */
    public static function obterInformacoesPlano($codigoBoramvno) {
        if (isset(self::$mapeamento[$codigoBoramvno])) {
            return self::$mapeamento[$codigoBoramvno];
        }
        return null;
    }
    
    /**
     * Verificar se um plano existe no mapeamento
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @return bool True se existe, false caso contrário
     */
    public static function planoExiste($codigoBoramvno) {
        return isset(self::$mapeamento[$codigoBoramvno]);
    }
    
    /**
     * Obter todos os códigos Boramvno disponíveis
     * 
     * @return array Array com todos os códigos Boramvno
     */
    public static function obterCodigosBoramvno() {
        return array_keys(self::$mapeamento);
    }
    
    /**
     * Obter todos os códigos IXC disponíveis
     * 
     * @return array Array com todos os códigos IXC
     */
    public static function obterCodigosIXC() {
        $codigos = [];
        foreach (self::$mapeamento as $plano) {
            $codigos[] = $plano['codigo_ixc'];
        }
        return $codigos;
    }
    
    /**
     * Buscar plano por código IXC
     * 
     * @param string $codigoIXC Código do plano no IXC
     * @return array|null Array com informações do plano ou null se não encontrado
     */
    public static function buscarPorCodigoIXC($codigoIXC) {
        foreach (self::$mapeamento as $codigoBoramvno => $plano) {
            if ($plano['codigo_ixc'] === $codigoIXC) {
                return array_merge(['codigo_boramvno' => $codigoBoramvno], $plano);
            }
        }
        return null;
    }
    
    /**
     * Obter estatísticas do mapeamento
     * 
     * @return array Array com estatísticas
     */
    public static function obterEstatisticas() {
        $total = count(self::$mapeamento);
        $tipos = [];
        $valores = [];
        
        foreach (self::$mapeamento as $plano) {
            $tipo = $plano['tipo_plano'];
            $tipos[$tipo] = isset($tipos[$tipo]) ? $tipos[$tipo] + 1 : 1;
            
            $valores[] = $plano['valor_plano'];
        }
        
        return [
            'total_planos' => $total,
            'tipos_disponiveis' => $tipos,
            'valor_medio' => $total > 0 ? array_sum($valores) / $total : 0,
            'valor_minimo' => $total > 0 ? min($valores) : 0,
            'valor_maximo' => $total > 0 ? max($valores) : 0
        ];
    }
    
    /**
     * Validar se o mapeamento está correto
     * 
     * @return array Array com erros encontrados
     */
    public static function validarMapeamento() {
        $erros = [];
        
        foreach (self::$mapeamento as $codigoBoramvno => $plano) {
            // Validar campos obrigatórios
            $camposObrigatorios = ['codigo_ixc', 'nome_plano', 'tipo_plano', 'valor_plano', 'duracao_plano'];
            
            foreach ($camposObrigatorios as $campo) {
                if (!isset($plano[$campo]) || empty($plano[$campo])) {
                    $erros[] = "Campo obrigatório '$campo' ausente ou vazio para o plano $codigoBoramvno";
                }
            }
            
            // Validar tipos de dados
            if (isset($plano['valor_plano']) && !is_numeric($plano['valor_plano'])) {
                $erros[] = "Valor do plano deve ser numérico para o plano $codigoBoramvno";
            }
            
            if (isset($plano['duracao_plano']) && !is_numeric($plano['duracao_plano'])) {
                $erros[] = "Duração do plano deve ser numérica para o plano $codigoBoramvno";
            }
            
            // Validar códigos únicos
            $codigosIXC = self::obterCodigosIXC();
            $ocorrencias = array_count_values($codigosIXC);
            if (isset($plano['codigo_ixc']) && $ocorrencias[$plano['codigo_ixc']] > 1) {
                $erros[] = "Código IXC duplicado: {$plano['codigo_ixc']}";
            }
        }
        
        return $erros;
    }
    
    /**
     * Exportar mapeamento para JSON
     * 
     * @return string JSON do mapeamento
     */
    public static function exportarParaJSON() {
        return json_encode(self::$mapeamento, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Importar mapeamento de JSON
     * 
     * @param string $json JSON com o mapeamento
     * @return bool True se importado com sucesso, false caso contrário
     */
    public static function importarDeJSON($json) {
        $dados = json_decode($json, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($dados)) {
            self::$mapeamento = $dados;
            return true;
        }
        return false;
    }
    
    /**
     * Adicionar novo plano ao mapeamento
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @param array $dadosPlano Dados do plano
     * @return bool True se adicionado com sucesso, false caso contrário
     */
    public static function adicionarPlano($codigoBoramvno, $dadosPlano) {
        if (empty($codigoBoramvno) || !is_array($dadosPlano)) {
            return false;
        }
        
        // Validar campos obrigatórios
        $camposObrigatorios = ['codigo_ixc', 'nome_plano', 'tipo_plano', 'valor_plano', 'duracao_plano'];
        foreach ($camposObrigatorios as $campo) {
            if (!isset($dadosPlano[$campo]) || empty($dadosPlano[$campo])) {
                return false;
            }
        }
        
        self::$mapeamento[$codigoBoramvno] = $dadosPlano;
        return true;
    }
    
    /**
     * Remover plano do mapeamento
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @return bool True se removido com sucesso, false caso contrário
     */
    public static function removerPlano($codigoBoramvno) {
        if (isset(self::$mapeamento[$codigoBoramvno])) {
            unset(self::$mapeamento[$codigoBoramvno]);
            return true;
        }
        return false;
    }
    
    /**
     * Atualizar plano existente
     * 
     * @param string $codigoBoramvno Código do plano na Boramvno
     * @param array $dadosPlano Novos dados do plano
     * @return bool True se atualizado com sucesso, false caso contrário
     */
    public static function atualizarPlano($codigoBoramvno, $dadosPlano) {
        if (!isset(self::$mapeamento[$codigoBoramvno]) || !is_array($dadosPlano)) {
            return false;
        }
        
        // Validar campos obrigatórios
        $camposObrigatorios = ['codigo_ixc', 'nome_plano', 'tipo_plano', 'valor_plano', 'duracao_plano'];
        foreach ($camposObrigatorios as $campo) {
            if (!isset($dadosPlano[$campo]) || empty($dadosPlano[$campo])) {
                return false;
            }
        }
        
        self::$mapeamento[$codigoBoramvno] = $dadosPlano;
        return true;
    }
}

/**
 * Funções de conveniência para uso global
 */

/**
 * Obter código IXC pelo código Boramvno (função global)
 * 
 * @param string $codigoBoramvno Código do plano na Boramvno
 * @return string|null Código do plano no IXC ou null se não encontrado
 */
function obterCodigoIXC($codigoBoramvno) {
    return MapeamentoPlanos::obterCodigoIXC($codigoBoramvno);
}

/**
 * Obter informações completas do plano (função global)
 * 
 * @param string $codigoBoramvno Código do plano na Boramvno
 * @return array|null Array com informações do plano ou null se não encontrado
 */
function obterInformacoesPlano($codigoBoramvno) {
    return MapeamentoPlanos::obterInformacoesPlano($codigoBoramvno);
}

/**
 * Verificar se um plano existe (função global)
 * 
 * @param string $codigoBoramvno Código do plano na Boramvno
 * @return bool True se existe, false caso contrário
 */
function planoExiste($codigoBoramvno) {
    return MapeamentoPlanos::planoExiste($codigoBoramvno);
}

/**
 * Obter mapeamento completo (função global)
 * 
 * @return array Array completo de mapeamento
 */
function obterMapeamentoCompleto() {
    return MapeamentoPlanos::obterMapeamentoCompleto();
}

/**
 * Buscar plano por ID Boramvno (função global)
 * 
 * @param string $planoBoramvno ID do plano na Boramvno
 * @return array Array com resultado da busca
 */
function buscarPlanoPorBoramvno($planoBoramvno) {
    return MapeamentoPlanos::buscarPlanoPorBoramvno($planoBoramvno);
}

// Log de inicialização
if (function_exists('logMessage')) {
    logMessage("Mapeamento de planos carregado com sucesso", "INFO");
}

// Verificar se há erros no mapeamento
$erros = MapeamentoPlanos::validarMapeamento();
if (!empty($erros)) {
    if (function_exists('logMessage')) {
        logMessage("Erros encontrados no mapeamento de planos: " . implode(', ', $erros), "WARNING");
    }
}

?>
