<?php
/**
 * Exemplo de uso da API IXCsoft
 * 
 * Este arquivo demonstra como usar a API programaticamente
 * para diferentes operações com clientes e outros recursos.
 */

// Incluir configuração e cliente da API
require_once(__DIR__ . DIRECTORY_SEPARATOR . 'config.php');
require_once(__DIR__ . DIRECTORY_SEPARATOR . 'WebserviceClient.php');

// Função para exibir resultados de forma organizada
function exibirResultado($titulo, $dados, $tipo = 'info') {
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "📋 $titulo\n";
    echo str_repeat("=", 60) . "\n";
    
    if (is_array($dados) || is_object($dados)) {
        echo json_encode($dados, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    } else {
        echo $dados;
    }
    echo "\n";
}

// Função para testar diferentes operações da API
function testarOperacoes() {
    try {
        // Obter configurações
        $host = getConfig('host');
        $token = getConfig('token');
        $selfSigned = getConfig('selfSigned');
        
        echo "🚀 Iniciando testes da API IXCsoft\n";
        echo "📍 Host: $host\n";
        echo "🔑 Token: " . substr($token, 0, 20) . "...\n";
        echo "🔒 Self-signed: " . ($selfSigned ? 'Sim' : 'Não') . "\n\n";
        
        // Criar instância da API
        $api = new IXCsoft\WebserviceClient($host, $token, $selfSigned);
        
        // 1. Teste de conexão básica
        echo "🔌 Testando conexão básica...\n";
        $api->get('cliente', ['page' => '1', 'rp' => '1']);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("✅ Conexão estabelecida", $response);
        
        // 2. Listar clientes (primeiros 5)
        echo "👥 Listando primeiros 5 clientes...\n";
        $api->get('cliente', [
            'page' => '1',
            'rp' => '5',
            'sortname' => 'cliente.id',
            'sortorder' => 'desc'
        ]);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("📋 Lista de clientes", $response);
        
        // 3. Buscar cliente por ID específico
        echo "🔍 Buscando cliente ID 1...\n";
        $api->get('cliente', [
            'qtype' => 'cliente.id',
            'query' => '1',
            'oper' => '=',
            'page' => '1',
            'rp' => '1'
        ]);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("👤 Cliente ID 1", $response);
        
        // 4. Buscar clientes por nome (exemplo)
        echo "🔍 Buscando clientes por nome...\n";
        $api->get('cliente', [
            'qtype' => 'cliente.nome',
            'query' => 'a',
            'oper' => 'LIKE',
            'page' => '1',
            'rp' => '3'
        ]);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("🔍 Clientes com nome contendo 'a'", $response);
        
        // 5. Testar paginação
        echo "📄 Testando paginação...\n";
        $api->get('cliente', [
            'page' => '2',
            'rp' => '3',
            'sortname' => 'cliente.id',
            'sortorder' => 'asc'
        ]);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("📄 Página 2 (3 registros)", $response);
        
        echo "\n🎉 Todos os testes foram executados com sucesso!\n";
        
    } catch (Exception $e) {
        echo "\n❌ Erro durante os testes: " . $e->getMessage() . "\n";
        echo "📁 Arquivo: " . $e->getFile() . "\n";
        echo "📍 Linha: " . $e->getLine() . "\n";
        
        // Log do erro
        logMessage("Erro nos testes: " . $e->getMessage(), "ERROR");
    }
}

// Função para demonstrar operações CRUD
function demonstrarCRUD() {
    try {
        echo "\n🔄 Demonstração de operações CRUD\n";
        echo str_repeat("-", 50) . "\n";
        
        $host = getConfig('host');
        $token = getConfig('token');
        $selfSigned = getConfig('selfSigned');
        
        $api = new IXCsoft\WebserviceClient($host, $token, $selfSigned);
        
        // READ - Ler dados
        echo "📖 READ: Lendo dados de clientes...\n";
        $api->get('cliente', ['page' => '1', 'rp' => '2']);
        $response = $api->getRespostaConteudo(false);
        exibirResultado("📖 Dados lidos", $response);
        
        // Exemplo de como seria um POST (criação)
        echo "📝 POST: Exemplo de criação (não executado)\n";
        echo "Para criar um cliente, você usaria:\n";
        echo "\$api->post('cliente', [\n";
        echo "    'nome' => 'João Silva',\n";
        echo "    'email' => 'joao@email.com',\n";
        echo "    'telefone' => '(11) 99999-9999'\n";
        echo "]);\n";
        
        // Exemplo de como seria um PUT (atualização)
        echo "\n📝 PUT: Exemplo de atualização (não executado)\n";
        echo "Para atualizar um cliente, você usaria:\n";
        echo "\$api->put('cliente', [\n";
        echo "    'nome' => 'João Silva Atualizado',\n";
        echo "    'email' => 'joao.novo@email.com'\n";
        echo "], 'ID_DO_CLIENTE');\n";
        
        // Exemplo de como seria um DELETE
        echo "\n🗑️ DELETE: Exemplo de exclusão (não executado)\n";
        echo "Para excluir um cliente, você usaria:\n";
        echo "\$api->delete('cliente', 'ID_DO_CLIENTE');\n";
        
    } catch (Exception $e) {
        echo "\n❌ Erro na demonstração CRUD: " . $e->getMessage() . "\n";
    }
}

// Função para mostrar estatísticas da API
function mostrarEstatisticas() {
    echo "\n📊 Estatísticas da API\n";
    echo str_repeat("-", 50) . "\n";
    
    echo "🌍 Ambiente: " . ENVIRONMENT . "\n";
    echo "🔗 Host: " . getConfig('host') . "\n";
    echo "🔧 Debug: " . (getConfig('debug') ? 'Ativado' : 'Desativado') . "\n";
    echo "📝 Log: " . (getConfig('log')['enabled'] ? 'Ativado' : 'Desativado') . "\n";
    echo "⏱️ Timeout: " . getConfig('timeout') . "s\n";
    echo "🔒 SSL Self-signed: " . (getConfig('selfSigned') ? 'Sim' : 'Não') . "\n";
    
    // Verificar arquivo de log
    if (getConfig('log')['enabled'] && file_exists(getConfig('log')['file'])) {
        $logSize = filesize(getConfig('log')['file']);
        echo "📁 Tamanho do log: " . number_format($logSize / 1024, 2) . " KB\n";
    }
}

// Menu principal
function mostrarMenu() {
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "🧪 TESTADOR DA API IXCSOFT\n";
    echo str_repeat("=", 60) . "\n";
    echo "1. 🚀 Executar todos os testes\n";
    echo "2. 🔄 Demonstração CRUD\n";
    echo "3. 📊 Mostrar estatísticas\n";
    echo "4. 🚪 Sair\n";
    echo str_repeat("=", 60) . "\n";
    echo "Escolha uma opção (1-4): ";
    
    $handle = fopen("php://stdin", "r");
    $opcao = trim(fgets($handle));
    fclose($handle);
    
    return $opcao;
}

// Execução principal
if (php_sapi_name() === 'cli') {
    // Modo CLI
    while (true) {
        $opcao = mostrarMenu();
        
        switch ($opcao) {
            case '1':
                testarOperacoes();
                break;
            case '2':
                demonstrarCRUD();
                break;
            case '3':
                mostrarEstatisticas();
                break;
            case '4':
                echo "\n👋 Saindo...\n";
                exit(0);
            default:
                echo "\n❌ Opção inválida. Tente novamente.\n";
        }
        
        echo "\nPressione Enter para continuar...";
        fgets(STDIN);
    }
} else {
    // Modo web - executar testes automaticamente
    echo "<!DOCTYPE html>
    <html lang='pt-BR'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>Exemplo de Uso da API</title>
        <style>
            body { font-family: monospace; margin: 20px; background: #f5f5f5; }
            .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 5px; }
            pre { background: #f8f9fa; padding: 15px; border-radius: 3px; overflow-x: auto; }
            .success { color: #28a745; }
            .error { color: #dc3545; }
            .info { color: #007bff; }
        </style>
    </head>
    <body>
        <div class='container'>
            <h1>🧪 Exemplo de Uso da API IXCsoft</h1>
            <p>Este arquivo demonstra como usar a API programaticamente.</p>
            <p>Para ver os resultados dos testes, execute este arquivo via CLI:</p>
            <pre>php exemplo-uso.php</pre>
            
            <h2>📋 Resultados dos Testes (Web)</h2>
            <pre>";
    
    // Executar testes em modo web
    ob_start();
    testarOperacoes();
    $output = ob_get_clean();
    
    // Converter quebras de linha para HTML
    echo htmlspecialchars($output);
    
    echo "</pre>
        </div>
    </body>
    </html>";
}
?>



