<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Consultar Planos - Sistema Boramvno</title>
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: #fff;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            color: white;
            padding: 30px;
            text-align: center;
            position: relative;
        }
        
        .header h1 {
            font-size: 2.5em;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .header p {
            font-size: 1.1em;
            opacity: 0.9;
        }
        
        .logo {
            position: absolute;
            top: 20px;
            left: 30px;
            width: 60px;
            height: 60px;
            background: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        
        .logo img {
            width: 40px;
            height: 40px;
            object-fit: contain;
        }
        
        .nav-buttons {
            position: absolute;
            top: 20px;
            right: 30px;
            display: flex;
            gap: 15px;
        }
        
        .nav-btn {
            background: rgba(255,255,255,0.2);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 25px;
            cursor: pointer;
            font-size: 0.9em;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .nav-btn:hover {
            background: rgba(255,255,255,0.3);
            transform: translateY(-2px);
        }
        
        .content {
            padding: 40px;
        }
        
        .search-section {
            background: #f8fafc;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            border: 2px solid #e2e8f0;
        }
        
        .search-section h2 {
            color: #1e293b;
            font-size: 1.5em;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #374151;
        }
        
        .form-input {
            width: 100%;
            max-width: 400px;
            padding: 12px 16px;
            border: 2px solid #d1d5db;
            border-radius: 10px;
            font-size: 1em;
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            outline: none;
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        
        .search-btn {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            border: none;
            padding: 14px 28px;
            border-radius: 10px;
            font-size: 1.1em;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 10px;
            box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
        }
        
        .search-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
        }
        
        .search-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .results-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            border: 2px solid #e2e8f0;
        }
        
        .results-section h2 {
            color: #1e293b;
            font-size: 1.5em;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .plans-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .plans-table th {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            color: white;
            padding: 15px;
            text-align: left;
            font-weight: 600;
            font-size: 1em;
        }
        
        .plans-table td {
            padding: 15px;
            border-bottom: 1px solid #e2e8f0;
            font-size: 0.95em;
        }
        
        .plans-table tr:nth-child(even) {
            background: #f8fafc;
        }
        
        .plans-table tr:hover {
            background: #f1f5f9;
        }
        
        .plans-table tr:last-child td {
            border-bottom: none;
        }
        
        .plan-id {
            font-family: monospace;
            background: #f1f5f9;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.9em;
            color: #374151;
        }
        
        .plan-name {
            font-weight: 600;
            color: #1e293b;
        }
        
        .plan-type {
            background: #dbeafe;
            color: #1e40af;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .plan-value {
            font-weight: 600;
            color: #059669;
            font-size: 1.1em;
        }
        
        .plan-duration {
            color: #6b7280;
            font-size: 0.9em;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #6b7280;
            font-size: 1.1em;
        }
        
        .loading .material-icons {
            font-size: 2em;
            color: #2563eb;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        .error {
            background: #fef2f2;
            border: 2px solid #fecaca;
            color: #dc2626;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            font-size: 1.1em;
        }
        
        .no-results {
            text-align: center;
            padding: 40px;
            color: #6b7280;
            font-size: 1.1em;
        }
        
        .info-box {
            background: #eff6ff;
            border: 2px solid #bfdbfe;
            color: #1e40af;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .info-box h3 {
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .info-box ul {
            margin-left: 20px;
            line-height: 1.6;
        }
        
        .back-btn {
            background: #6b7280;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 1em;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 20px;
        }
        
        .back-btn:hover {
            background: #4b5563;
            transform: translateY(-1px);
        }
        
        @media (max-width: 768px) {
            .container {
                margin: 10px;
                border-radius: 15px;
            }
            
            .header {
                padding: 20px;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .content {
                padding: 20px;
            }
            
            .plans-table {
                font-size: 0.9em;
            }
            
            .plans-table th,
            .plans-table td {
                padding: 10px 8px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="logo">
                <img src="chip.png" alt="Logo" onerror="this.style.display='none'">
            </div>
            <h1>Consultar Planos</h1>
            <p>Consulta à API /api/Plan/Activation para visualizar planos disponíveis</p>
            <div class="nav-buttons">
                <a href="index.php" class="nav-btn">
                    <span class="material-icons">home</span>
                    Início
                </a>
                <a href="cadastro.php" class="nav-btn">
                    <span class="material-icons">person_add</span>
                    Cadastro
                </a>
                <a href="linhas.php" class="nav-btn">
                    <span class="material-icons">sim_card</span>
                    Linhas
                </a>
                <a href="consumos.php" class="nav-btn">
                    <span class="material-icons">analytics</span>
                    Consumos
                </a>
                <a href="logout.php" class="nav-btn">
                    <span class="material-icons">logout</span>
                    Sair
                </a>
            </div>
        </div>
        
        <div class="content">
            <a href="index.php" class="back-btn">
                <span class="material-icons">arrow_back</span>
                Voltar ao Início
            </a>
            
            <div class="info-box">
                <h3><span class="material-icons">info</span>Informações sobre a API</h3>
                <ul>
                    <li><strong>Endpoint:</strong> /api/Plan/Activation</li>
                    <li><strong>Método:</strong> GET</li>
                    <li><strong>Parâmetros obrigatórios:</strong> Type (Pos) e ICCID</li>
                    <li><strong>Retorna:</strong> Lista de planos disponíveis para ativação</li>
                    <li><strong>Campos retornados:</strong> idPlanExternal, name, type, label, value, durationTime</li>
                </ul>
            </div>
            
            <div class="search-section">
                <h2><span class="material-icons">search</span>Consultar Planos Disponíveis</h2>
                
                <div class="form-group">
                    <label for="iccidInput">ICCID do Chip:</label>
                    <input type="text" id="iccidInput" class="form-input" placeholder="Ex: 89551807400453032859" maxlength="20">
                </div>
                
                <button id="consultarPlanosBtn" class="search-btn">
                    <span class="material-icons">search</span>
                    Consultar Planos
                </button>
            </div>
            
            <div class="results-section">
                <h2><span class="material-icons">list</span>Planos Disponíveis</h2>
                <div id="planosResult">
                    <div class="no-results">
                        Digite um ICCID e clique em "Consultar Planos" para ver os planos disponíveis
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const consultarPlanosBtn = document.getElementById('consultarPlanosBtn');
            const iccidInput = document.getElementById('iccidInput');
            const planosResult = document.getElementById('planosResult');
            
            consultarPlanosBtn.addEventListener('click', function() {
                const iccid = iccidInput.value.trim();
                
                if (!iccid) {
                    alert('Por favor, digite um ICCID válido.');
                    return;
                }
                
                // Validar formato do ICCID (deve ter pelo menos 19 dígitos)
                if (iccid.length < 19) {
                    alert('ICCID deve ter pelo menos 19 dígitos.');
                    return;
                }
                
                // Desabilitar botão e mostrar loading
                consultarPlanosBtn.disabled = true;
                consultarPlanosBtn.innerHTML = '<span class="material-icons">hourglass_top</span>Consultando...';
                
                // Mostrar loading
                planosResult.innerHTML = `
                    <div class="loading">
                        <span class="material-icons">hourglass_top</span>
                        <br>Consultando planos disponíveis...
                    </div>
                `;
                
                // Fazer requisição para a API
                const url = 'proxy-chip.php?rota=' + encodeURIComponent(`/api/Plan/Activation?Type=Pos&Iccid=${iccid}`);
                
                fetch(url)
                    .then(response => response.json())
                    .then(data => {
                        // Reabilitar botão
                        consultarPlanosBtn.disabled = false;
                        consultarPlanosBtn.innerHTML = '<span class="material-icons">search</span>Consultar Planos';
                        
                        // Verificar se há dados
                        if (data.error) {
                            planosResult.innerHTML = `
                                <div class="error">
                                    <strong>Erro na consulta:</strong><br>
                                    ${data.error}<br>
                                    <small>HTTP Code: ${data.http_code || 'N/A'}</small>
                                </div>
                            `;
                            return;
                        }
                        
                        // Extrair planos dos dados
                        let planos = data;
                        if (!Array.isArray(planos) && data.data) {
                            planos = data.data;
                        }
                        
                        if (!Array.isArray(planos) || planos.length === 0) {
                            planosResult.innerHTML = `
                                <div class="no-results">
                                    <span class="material-icons" style="font-size: 3em; color: #9ca3af; margin-bottom: 10px;">inbox</span>
                                    <br>Nenhum plano encontrado para este ICCID.
                                </div>
                            `;
                            return;
                        }
                        
                        // Criar tabela com os planos
                        let tableHTML = `
                            <table class="plans-table">
                                <thead>
                                    <tr>
                                        <th>Código do Plano</th>
                                        <th>Nome do Plano</th>
                                        <th>Tipo</th>
                                        <th>Valor</th>
                                        <th>Duração</th>
                                        <th>Descrição</th>
                                    </tr>
                                </thead>
                                <tbody>
                        `;
                        
                        planos.forEach((plano, index) => {
                            // Extrair informações do plano
                            const planId = plano.idPlanExternal || plano.planId || plano.id || 'N/A';
                            const planName = plano.name || plano.planName || 'Nome não informado';
                            const planType = plano.type || 'Tipo não informado';
                            const planValue = plano.value || plano.price || 0;
                            const planDuration = plano.durationTime || 'N/A';
                            const planDescription = plano.description || plano.label || 'Descrição não informada';
                            
                            // Formatar valor
                            let formattedValue = 'R$ 0,00';
                            if (planValue && !isNaN(planValue)) {
                                formattedValue = `R$ ${(parseFloat(planValue) / 100).toFixed(2).replace('.', ',')}`;
                            }
                            
                            // Formatar duração
                            let formattedDuration = planDuration;
                            if (planDuration && !isNaN(planDuration)) {
                                formattedDuration = `${planDuration} dias`;
                            }
                            
                            tableHTML += `
                                <tr>
                                    <td><span class="plan-id">${planId}</span></td>
                                    <td><span class="plan-name">${planName}</span></td>
                                    <td><span class="plan-type">${planType}</span></td>
                                    <td><span class="plan-value">${formattedValue}</span></td>
                                    <td><span class="plan-duration">${formattedDuration}</span></td>
                                    <td>${planDescription}</td>
                                </tr>
                            `;
                        });
                        
                        tableHTML += `
                                </tbody>
                            </table>
                            <div style="margin-top: 20px; text-align: center; color: #6b7280; font-size: 0.9em;">
                                <strong>Total de planos encontrados:</strong> ${planos.length}
                            </div>
                        `;
                        
                        planosResult.innerHTML = tableHTML;
                        
                        // Log dos dados para debugging
                        console.log('📊 Planos recebidos da API:', planos);
                        console.log('📊 Quantidade de planos:', planos.length);
                        
                    })
                    .catch(error => {
                        // Reabilitar botão
                        consultarPlanosBtn.disabled = false;
                        consultarPlanosBtn.innerHTML = '<span class="material-icons">search</span>Consultar Planos';
                        
                        console.error('❌ Erro na consulta:', error);
                        planosResult.innerHTML = `
                            <div class="error">
                                <strong>Erro na consulta:</strong><br>
                                ${error.message}<br>
                                <small>Verifique o console para mais detalhes</small>
                            </div>
                        `;
                    });
            });
            
            // Permitir consulta ao pressionar Enter
            iccidInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    consultarPlanosBtn.click();
                }
            });
            
            // Focar no input ao carregar a página
            iccidInput.focus();
        });
    </script>
</body>
</html>
