#!/bin/bash

echo "🚀 Iniciando configuração do servidor remoto..."
echo "================================================"

# 1. Reverter sources.list para Debian 10 (Buster)
echo "📋 Passo 1: Revertendo sources.list para Debian 10..."
sed -i 's/bookworm/buster/g' /etc/apt/sources.list
echo "✅ Sources.list revertido para Buster"

# 2. Limpar e atualizar pacotes
echo "📋 Passo 2: Limpando e atualizando pacotes..."
apt-get clean
apt-get update --fix-missing
echo "✅ Pacotes atualizados"

# 3. Instalar PHP 7.3 e extensões necessárias
echo "📋 Passo 3: Instalando PHP 7.3 e extensões..."
apt-get install -y --reinstall php7.3-common php7.3-cli libapache2-mod-php7.3 php7.3-curl php7.3-mysql php7.3-json php7.3-opcache
echo "✅ PHP 7.3 instalado"

# 4. Configurar Apache
echo "📋 Passo 4: Configurando Apache..."
a2enmod rewrite
a2enmod headers

# Configurar DocumentRoot e AllowOverride
cat > /etc/apache2/sites-available/000-default.conf << 'EOF'
<VirtualHost *:80>
    ServerAdmin webmaster@localhost
    DocumentRoot /var/www
    
    <Directory /var/www/>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/error.log
    CustomLog ${APACHE_LOG_DIR}/access.log combined
</VirtualHost>
EOF

echo "✅ Apache configurado"

# 5. Criar diretório do sistema
echo "📋 Passo 5: Criando diretório do sistema..."
mkdir -p /var/www/sistema
mkdir -p /var/www/sistema/logs
chown -R www-data:www-data /var/www/sistema
chmod -R 755 /var/www/sistema
echo "✅ Diretórios criados"

# 6. Criar link simbólico
echo "📋 Passo 6: Criando link simbólico..."
ln -sf /var/www/sistema /var/www/html/sistema
echo "✅ Link simbólico criado"

# 7. Configurar PHP
echo "📋 Passo 7: Configurando PHP..."
cat > /etc/php/7.3/apache2/conf.d/99-custom.ini << 'EOF'
; Configurações customizadas para o sistema
memory_limit = 256M
max_execution_time = 300
upload_max_filesize = 50M
post_max_size = 50M
display_errors = Off
log_errors = On
error_log = /var/www/sistema/logs/php_errors.log
date.timezone = America/Sao_Paulo
EOF

echo "✅ PHP configurado"

# 8. Reiniciar serviços
echo "📋 Passo 8: Reiniciando serviços..."
systemctl restart apache2
echo "✅ Apache reiniciado"

# 9. Verificar status
echo "📋 Passo 9: Verificando status..."
echo "🔍 Versão do PHP:"
php -v

echo "🔍 Status do Apache:"
systemctl status apache2 --no-pager -l

echo "🔍 Módulos Apache ativos:"
apache2ctl -M | grep -E "(rewrite|headers)"

echo "🔍 Permissões do diretório:"
ls -la /var/www/sistema

echo "================================================"
echo "🎉 Configuração do servidor concluída!"
echo "🌐 URL: http://192.168.2.16/sistema"
echo "📁 Diretório: /var/www/sistema"
echo "📝 Logs: /var/www/sistema/logs"
echo ""
echo "📋 Próximos passos:"
echo "1. Fazer upload dos arquivos do sistema"
echo "2. Configurar banco de dados MySQL"
echo "3. Testar o sistema"
echo "================================================"
